// 全局变量
let movieData = null;

// 可以删除不再使用的downloadSubtitle函数，因为我们将使用<a>标签直接下载
let groupItems = [];
let currentVideoIndex = 0;
let customFieldData = {
    storeSiteAlias: '',
    customClassText: ''
};
let serverConfig = {
    ip: '127.0.0.1',
    port: '5280,5281'
};
let clientId = localStorage.getItem('jeenpiClientID') || '';

// 检测系统类型
function detectOS() {
    const userAgent = navigator.userAgent || navigator.vendor || window.opera;
    
    if (/Mac OS X/i.test(userAgent) && !/iPhone|iPad|iPod/i.test(userAgent)) {
        return 'macOS';
    } else if (/iPhone|iPad|iPod/i.test(userAgent)) {
        return 'iOS';
    } else if (/Android/i.test(userAgent)) {
        return 'Android';
    } else if (/Windows/i.test(userAgent)) {
        return 'Windows';
    } else if (/Linux/i.test(userAgent)) {
        return 'Linux';
    }
    return 'Unknown';
}

const currentOS = detectOS();

// DOM元素
const errorDialog = document.getElementById('errorDialog');
const errorDialogTitle = document.getElementById('errorDialogTitle');
const errorDialogMessage = document.getElementById('errorDialogMessage');
const closeErrorDialog = document.getElementById('closeErrorDialog');
const playerDialog = document.getElementById('playerDialog');
const playerDialogHint = document.getElementById('playerDialogHint');
const playerOptions = document.getElementById('playerOptions');
const useInternalPlayer = document.getElementById('useInternalPlayer');
const closePlayerDialog = document.getElementById('closePlayerDialog');
const loadingIndicator = document.getElementById('loadingIndicator');
const movieDetailContent = document.getElementById('movieDetailContent');
const errorState = document.getElementById('errorState');
const backBtn = document.getElementById('backBtn');
const homeBtn = document.getElementById('homeBtn');
const retryBtn = document.getElementById('retryBtn');
const playLoading = document.getElementById('playLoading');
const videoPlayerContainer = document.getElementById('videoPlayerContainer');
const moviePlayer = document.getElementById('moviePlayer');
const playerLoading = document.getElementById('playerLoading');

// 初始化页面
function initPage() {
    hideErrorDialog();
    hidePlayerDialog();
    
    const savedConfig = localStorage.getItem('jeenpiServerConfig');
    if (savedConfig) {
        serverConfig = JSON.parse(savedConfig);
    }
    
    clientId = localStorage.getItem('ClientID') || clientId;
    
    closeErrorDialog.addEventListener('click', hideErrorDialog);
    errorDialog.addEventListener('click', (e) => {
        if (e.target === errorDialog) hideErrorDialog();
    });
    
    closePlayerDialog.addEventListener('click', hidePlayerDialog);
    useInternalPlayer.addEventListener('click', () => {
        hidePlayerDialog();
        playMovieInternal();
    });
    
    playerDialog.addEventListener('click', (e) => {
        if (e.target === playerDialog) hidePlayerDialog();
    });
    
    // 从localStorage获取电影数据和自动播放标志
    const selectedMovie = localStorage.getItem('selectedMovie');
    const autoPlayStr = localStorage.getItem('movieAutoPlay');
    const autoPlay = autoPlayStr === 'true';
    
    // 获取数据后立即从localStorage中移除，避免重复使用
    localStorage.removeItem('selectedMovie');
    localStorage.removeItem('movieAutoPlay');
    
    if (selectedMovie) {
        try {
            movieData = JSON.parse(selectedMovie);
            fetchCustomFieldValues().then(() => {
                // 获取导演和演员头像
                return fetchCelebrityAvatars();
            }).then(() => {
                if (movieData.FileData?.GroupID && movieData.FileData.GroupID !== '') {
                    fetchGroupItems().then(() => {
                        if (groupItems.length > 0) {
                            currentVideoIndex = 0;
                            movieData = groupItems[0];
                            fetchCustomFieldValues().then(() => {
                                // 获取导演和演员头像
                                return fetchCelebrityAvatars();
                            }).then(() => {
                                renderMovieDetail();
                                
                                // 如果需要自动播放，在渲染完成后触发播放
                                if (autoPlay) {
                                    setTimeout(() => {
                                        const playBtn = document.getElementById('playBtn');
                                        if (playBtn) {
                                            playBtn.click();
                                        }
                                    }, 100); // 短暂延迟确保DOM已完全渲染
                                }
                            });
                        } else {
                            renderMovieDetail();
                            
                            // 如果需要自动播放，在渲染完成后触发播放
                            if (autoPlay) {
                                setTimeout(() => {
                                    const playBtn = document.getElementById('playBtn');
                                    if (playBtn) {
                                        playBtn.click();
                                    }
                                }, 100); // 短暂延迟确保DOM已完全渲染
                            }
                        }
                    }).catch(() => {
                        renderMovieDetail();
                        
                        // 如果需要自动播放，在渲染完成后触发播放
                        if (autoPlay) {
                            setTimeout(() => {
                                const playBtn = document.getElementById('playBtn');
                                if (playBtn) {
                                    playBtn.click();
                                }
                            }, 100); // 短暂延迟确保DOM已完全渲染
                        }
                    });
                } else {
                    renderMovieDetail();
                    
                    // 如果需要自动播放，在渲染完成后触发播放
                    if (autoPlay) {
                        setTimeout(() => {
                            const playBtn = document.getElementById('playBtn');
                            if (playBtn) {
                                playBtn.click();
                            }
                        }, 100); // 短暂延迟确保DOM已完全渲染
                    }
                }
            }).catch(() => {
                renderMovieDetail();
                
                // 如果需要自动播放，在渲染完成后触发播放
                if (autoPlay) {
                    setTimeout(() => {
                        const playBtn = document.getElementById('playBtn');
                        if (playBtn) {
                            playBtn.click();
                        }
                    }, 100); // 短暂延迟确保DOM已完全渲染
                }
            });
        } catch (error) {
            showErrorState();
        }
    } else {
        // 如果localStorage中没有数据，显示错误状态
        showErrorState();
    }
    
    backBtn.addEventListener('click', () => window.history.back());
    homeBtn.addEventListener('click', () => window.location.href = 'index.html');
    retryBtn.addEventListener('click', () => {
        // 直接返回首页，让用户重新选择影片
        window.location.href = 'index.html';
    });
    
    moviePlayer.addEventListener('loadedmetadata', () => {
        playerLoading.classList.add('hidden');
    });
    
    moviePlayer.addEventListener('error', (e) => {
        playerLoading.classList.add('hidden');
        if (videoPlayerContainer.classList.contains('hidden') === false) {
            showErrorDialog('播放失败', getVideoErrorText(e.target.error));
        }
    });
}

function showErrorDialog(title, message) {
    errorDialogTitle.textContent = title;
    errorDialogMessage.textContent = message;
    errorDialog.classList.add('show');
}

function hideErrorDialog() {
    errorDialog.classList.remove('show');
}

function showPlayerDialog(videoUrl) {
    let osName = '';
    let players = [];
    
    if (currentOS === 'macOS') {
        osName = 'macOS';
        players = [
            { name: 'IINA', scheme: 'iina://weblink?url=', icon: 'fa-film' }
        ];
    } else if (currentOS === 'iOS') {
        osName = 'iOS';
        players = [
            { name: 'VLC', scheme: 'vlc-x-callback://x-callback-url/stream?url=', icon: 'fa-play-circle' },
            { name: 'nPlayer', scheme: 'nplayer-', icon: 'fa-film' }
        ];
    } else if (currentOS === 'Android') {
        osName = 'Android';
        players = [
            { name: 'VLC', scheme: 'vlc://', icon: 'fa-play-circle' },
            { name: 'MX Player', scheme: 'intent:', icon: 'fa-film' }
        ];
    } else if (currentOS === 'Windows') {
            osName = 'Windows';
            players = [
                { name: 'PotPlayer', scheme: 'potplayer://', icon: 'fa-play-circle' },
                { name: 'VLC', scheme: 'vlc-x-callback://x-callback-url/stream?url=', icon: 'fa-play-circle' }
            ];
        } else {
        // 其他系统默认添加PotPlayer和VLC
        osName = '其他操作系统';
        players = [
            { name: 'PotPlayer', scheme: 'potplayer://', icon: 'fa-play-circle' },
            { name: 'VLC', scheme: 'vlc-x-callback://x-callback-url/stream?url=', icon: 'fa-play-circle' }
        ];
    }
    
    playerDialogHint.textContent = `检测到您使用的是 ${osName} 系统，推荐使用外部播放器获得更好的体验`;
    
    // 获取字幕URL（如果存在）
    let subtitleUrl = '';
    let subtitleExtension = '';
    if (movieData.SubtitleURL || movieData.Subtitles) {
        const resourcePort = serverConfig.port.split(',')[1] || '5281';
        const subtitlePath = movieData.SubtitleURL || movieData.Subtitles;
        
        if (subtitlePath.startsWith('http')) {
            subtitleUrl = subtitlePath;
        } else {
            subtitleUrl = `http://${serverConfig.ip}:${resourcePort}${subtitlePath}`;
        }
        
        // 提取字幕文件扩展名
        const lastDotIndex = subtitlePath.lastIndexOf('.');
        if (lastDotIndex > -1) {
            subtitleExtension = subtitlePath.substring(lastDotIndex);
        }
    }
    

    
    // 对于Android设备，使用更简单直接的HTML实现
    if (currentOS === 'Android') {
        playerOptions.innerHTML = `
            <div class="space-y-3">
                <a href="vlc://${encodeURIComponent(videoUrl)}" class="player-option w-full py-3 px-4 bg-white border-2 border-gray-200 rounded-lg hover:border-primary hover:bg-blue-50 transition-all text-left flex items-center block">
                    <i class="fa fa-play-circle text-primary text-2xl mr-3"></i>
                    <div class="flex-1">
                        <div class="font-medium text-gray-900">VLC</div>
                        <div class="text-xs text-gray-500">使用VLC打开播放</div>
                    </div>
                    <i class="fa fa-chevron-right text-gray-400"></i>
                </a>
                <a href="intent:${encodeURIComponent(videoUrl)}#Intent;action=android.intent.action.VIEW;type=video/*;end" class="player-option w-full py-3 px-4 bg-white border-2 border-gray-200 rounded-lg hover:border-primary hover:bg-blue-50 transition-all text-left flex items-center block">
                    <i class="fa fa-film text-primary text-2xl mr-3"></i>
                    <div class="flex-1">
                        <div class="font-medium text-gray-900">MX Player</div>
                        <div class="text-xs text-gray-500">使用MX Player打开播放</div>
                    </div>
                    <i class="fa fa-chevron-right text-gray-400"></i>
                </a>
                ${subtitleUrl ? `<a href="${subtitleUrl}" download target="_blank" class="player-option w-full py-2 px-3 bg-white border-1 border-gray-200 rounded-lg hover:border-primary hover:bg-blue-50 transition-all text-left flex items-center justify-between block">
                    <div class="flex items-center">
                        <i class="fa fa-download text-green-500 text-lg mr-2"></i>
                        <span class="text-xs text-gray-500">包含字幕文件(${subtitleExtension || '.ass'})</span>
                    </div>
                    <span class="font-medium text-gray-600 text-sm">下载字幕</span>
                </a>` : ''}

            </div>
        `;
        

    } else {
        // 构建播放器选项HTML
        let playerOptionsHtml = '';
        
        // 添加播放器选项
        playerOptionsHtml += players.map(player => `
            <button class="player-option w-full py-3 px-4 bg-white border-2 border-gray-200 rounded-lg hover:border-primary hover:bg-blue-50 transition-all text-left flex items-center" data-player="${player.name}" data-scheme="${player.scheme}">
                <i class="fa ${player.icon} text-primary text-2xl mr-3"></i>
                <div class="flex-1">
                    <div class="font-medium text-gray-900">${player.name}</div>
                    <div class="text-xs text-gray-500">使用 ${player.name} 打开播放</div>
                </div>
                <i class="fa fa-chevron-right text-gray-400"></i>
            </button>
        `).join('');
        
        // 如果有字幕，添加下载按钮
            if (subtitleUrl) {
                playerOptionsHtml += `
                 <a href="${subtitleUrl}" download target="_blank" class="player-option w-full py-2 px-3 bg-white border-1 border-gray-200 rounded-lg hover:border-primary hover:bg-blue-50 transition-all text-left flex items-center justify-between">
                    <div class="flex items-center">
                        <i class="fa fa-download text-green-500 text-lg mr-2"></i>
                        <span class="text-xs text-gray-500">包含字幕文件(${subtitleExtension || '.ass'})</span>
                    </div>
                    <span class="font-medium text-gray-600 text-sm">下载字幕</span>
                </a>
            `;
            }
        
        playerOptions.innerHTML = playerOptionsHtml;
        

        
        // 为播放器选项添加事件监听
        document.querySelectorAll('.player-option').forEach(btn => {
            btn.addEventListener('click', function() {
                const playerName = this.dataset.player;
                const scheme = this.dataset.scheme;
                openExternalPlayer(videoUrl, scheme, playerName);
            });
        });
    }
    
    playerDialog.classList.add('show');
}

function hidePlayerDialog() {
    playerDialog.classList.remove('show');
}

function openExternalPlayer(videoUrl, scheme, playerName) {
    let finalUrl = '';
    
    // 获取字幕URL（如果存在）
    let subtitleUrl = '';
    let subtitleExtension = '';
    if (movieData.SubtitleURL || movieData.Subtitles) {
        const resourcePort = serverConfig.port.split(',')[1] || '5281';
        const subtitlePath = movieData.SubtitleURL || movieData.Subtitles;
        
        if (subtitlePath.startsWith('http')) {
            subtitleUrl = subtitlePath;
        } else {
            subtitleUrl = `http://${serverConfig.ip}:${resourcePort}${subtitlePath}`;
        }
        
        // 提取字幕文件扩展名
        const lastDotIndex = subtitlePath.lastIndexOf('.');
        if (lastDotIndex > -1) {
            subtitleExtension = subtitlePath.substring(lastDotIndex);
        }
    }
    
    // 根据不同播放器构建 URL Scheme - 只保留视频URL和字幕URL
    if (currentOS === 'Android' && playerName === 'MX Player') {
        // MX Player - 简化格式，只保留视频URL
        const packageName = "com.mxtech.videoplayer.ad";
        finalUrl = `intent:${encodeURI(videoUrl)}#Intent;package=${packageName};end`;
    } else if (playerName === 'VLC') {
        // VLC - 不同平台的简化格式
        if (currentOS === 'macOS') {
            // macOS下使用标准的vlc://格式
            finalUrl = `vlc://${encodeURI(videoUrl)}`;
            if (subtitleUrl) {
                finalUrl += `?subtitles=${encodeURI(subtitleUrl)}`;
            }
        } else if (currentOS === 'Android') {
            // Android上VLC使用Intent格式
            finalUrl = `intent:${encodeURI(videoUrl)}#Intent;package=org.videolan.vlc;type=video/*;end`;
        } else {
            // 其他平台使用vlc-x-callback格式
            finalUrl = `vlc-x-callback://x-callback-url/stream?url=${encodeURI(videoUrl)}`;
            if (subtitleUrl) {
                finalUrl += `&sub=${encodeURI(subtitleUrl)}`;
            }
        }
    } else if (playerName === 'PotPlayer') {
        // PotPlayer - 确保正确添加字幕URL，使用PotPlayer特定的参数格式
        finalUrl = `potplayer://${encodeURI(videoUrl)}`;
        if (subtitleUrl) {
            // PotPlayer使用的是空格+参数名=值的格式
            finalUrl += ' /sub=' + encodeURI(subtitleUrl);
        }
    } else if (currentOS === 'iOS' && playerName === 'nPlayer') {
        // nPlayer - 简化格式
        finalUrl = `nplayer-${encodeURI(videoUrl)}`;
        if (subtitleUrl) {
            const separator = videoUrl.includes('?') ? '&' : '?';
            finalUrl = `nplayer-${encodeURI(videoUrl)}${separator}subtitle=${encodeURI(subtitleUrl)}`;
        }
    } else if (playerName === 'IINA') {
        // IINA播放器 - 简化格式
        finalUrl = `iina://weblink?url=${encodeURI(videoUrl)}`;
        if (subtitleUrl) {
            finalUrl += `&mpv_sub-files=${encodeURI(subtitleUrl)}`;
        }
    } else {
        // 其他播放器默认处理
        finalUrl = scheme + encodeURI(videoUrl);
        if (subtitleUrl) {
            // 尝试添加字幕参数（具体格式可能需要根据播放器调整）
            finalUrl += '?subtitle=' + encodeURI(subtitleUrl);
        }
    }
    
    // 尝试打开外部播放器
    
    // 不再需要为macOS上的VLC准备备用URL格式
    
    // 改进的检测机制
    let hasLeftPage = false;
    let hasBlurred = false;
    const startTime = Date.now();
    let checkTimer = null;
    
    // 监听页面可见性变化（App切换）
    const visibilityHandler = () => {
        if (document.hidden) {
            hasLeftPage = true;
        }
    };
    document.addEventListener('visibilitychange', visibilityHandler);
    
    // 监听页面失焦（App切换）
    const blurHandler = () => {
        hasBlurred = true;
    };
    window.addEventListener('blur', blurHandler);
    
    // 监听页面重新获得焦点
    const focusHandler = () => {
        // 如果之前已经失焦或隐藏，说明确实打开了外部应用
        if (hasBlurred || hasLeftPage) {
            clearTimeout(checkTimer);
            cleanup();
            hidePlayerDialog();
            // 不显示任何提示，因为已经成功

        }
    };
    window.addEventListener('focus', focusHandler);
    
    // 清理函数
    const cleanup = () => {
        document.removeEventListener('visibilitychange', visibilityHandler);
        window.removeEventListener('blur', blurHandler);
        window.removeEventListener('focus', focusHandler);
        if (link && link.parentNode) {
            document.body.removeChild(link);
        }
        // 备用链接会在各自的try/catch块中被清理
    };
    
    // 尝试打开 - 针对PotPlayer特殊处理以避免竖线符号被编码
    try {
        // 对于PotPlayer，我们需要特殊处理竖线符号
        if (playerName === 'PotPlayer') {

            
            // 使用更简单的方式：直接设置window.location
            window.location.href = finalUrl;
        } else {
            // 其他播放器使用常规方法

            window.open(finalUrl, '_self');
        }
    } catch (e) {
        // 备用方案：使用a标签点击
        const link = document.createElement('a');
        link.href = finalUrl;
        link.style.display = 'none';
        document.body.appendChild(link);
        link.click();
        setTimeout(() => {
            if (link.parentNode) {
                document.body.removeChild(link);
            }
        }, 100);
    }
    
    // 使用更长的超时时间进行检测（3秒）
    checkTimer = setTimeout(() => {
        const elapsedTime = Date.now() - startTime;
        
        // 如果页面失焦或隐藏过，认为可能成功
        if (hasLeftPage || hasBlurred) {

            cleanup();
            hidePlayerDialog();
            // 成功时不显示提示，用户体验更好
        } else {
            // 其他情况，显示错误

            cleanup();
            hidePlayerDialog();
            showErrorDialog(
                '无法打开播放器', 
                `未能打开 ${playerName} 应用。请确认：\n1. 已安装 ${playerName} 应用\n2. 已授予必要的权限\n\n您可以选择使用内置播放器播放。`
            );
        }
    }, 3000);
}

function getVideoErrorText(error) {
    switch(error.code) {
        case error.MEDIA_ERR_ABORTED:
            return "视频加载被中止，请重试。";
        case error.MEDIA_ERR_NETWORK:
            return "网络错误导致视频加载失败，请检查网络连接后重试。";
        case error.MEDIA_ERR_DECODE:
            return "视频文件损坏或格式不支持，无法播放。";
        case error.MEDIA_ERR_SRC_NOT_SUPPORTED:
            return "影片文件不存在或无法访问，无法播放！";
        default:
            return "播放过程中发生未知错误，请尝试其他影片。";
    }
}

async function playMovie() {
    if (!movieData || !movieData.FileData) {
        showErrorDialog('播放失败', '无法获取影片信息，无法播放！');
        return;
    }
    
    playLoading.classList.remove('hidden');
    playLoading.classList.add('flex');
    
    try {
        const pid = movieData.FileData.AID;
        const fileName = movieData.FileData.FileName;
        const storeSite = movieData.FileData.StoreSite;
        
        if (!pid || !fileName || !storeSite) {
            throw new Error('影片信息不完整，无法播放！');
        }
        
        const paramsDataObj = {
            "PID": pid,
            "FileName": fileName,
            "StoreSite": storeSite
        };
        
        const jsonStr = JSON.stringify(paramsDataObj);
        const base64Params = base64Encode(jsonStr);
        
        const formData = new FormData();
        formData.append('AppKey', 'B623D8449570');
        formData.append('ClientID', clientId);
        formData.append('ParamsData', base64Params);
        
        const apiPort = serverConfig.port.split(',')[0] || '5280';
        const url = `http://${serverConfig.ip}:${apiPort}/JeenpiAPI/V1/PlayMovieDirectly`;
        
        const response = await fetch(url, {
            method: 'POST',
            body: formData
        });
        
        const encodedResponse = await response.text();
        const data = JSON.parse(encodedResponse);
        
        playLoading.classList.add('hidden');
        playLoading.classList.remove('flex');
        
        if (data.code && data.code > 0) {
            let errorMsg = data.msg || `播放请求失败，错误代码: ${data.code}`;
            
            if (data.code === 404) {
                errorMsg = "影片文件不存在，无法播放！";
            } else if (data.code === 403) {
                errorMsg = "您没有权限播放此影片。";
            } else if (data.code >= 500) {
                errorMsg = "服务器错误，请稍后再试。";
            }
            
            throw new Error(errorMsg);
        }
        
        if (data.Data && data.Data.URLFileName) {
            let videoUrl = `http://${serverConfig.ip}:${serverConfig.port.split(',')[1] || '5281'}${data.Data.URLFileName}`;
            
            // 保存字幕URL（从Subtitles字段获取）
            if (data.Data.Subtitles) {
                movieData.SubtitleURL = data.Data.Subtitles;
            }
            
            // 处理.strm文件
            videoUrl = await processStrmFile(videoUrl);
            
            // 对所有系统都显示播放器选择对话框，让用户自行选择使用哪个播放器
            showPlayerDialog(videoUrl);
        } else {
            throw new Error('未获取到有效的播放链接，无法播放！');
        }
    } catch (error) {
        playLoading.classList.add('hidden');
        playLoading.classList.remove('flex');
        showErrorDialog('播放失败', error.message);
    }
}

async function playMovieInternal() {
    if (!movieData || !movieData.FileData) {
        showErrorDialog('播放失败', '无法获取影片信息，无法播放！');
        return;
    }
    
    playLoading.classList.remove('hidden');
    playLoading.classList.add('flex');
    
    try {
        const pid = movieData.FileData.AID;
        const fileName = movieData.FileData.FileName;
        const storeSite = movieData.FileData.StoreSite;
        
        if (!pid || !fileName || !storeSite) {
            throw new Error('影片信息不完整，无法播放！');
        }
        
        const paramsDataObj = {
            "PID": pid,
            "FileName": fileName,
            "StoreSite": storeSite
        };
        
        const jsonStr = JSON.stringify(paramsDataObj);
        const base64Params = base64Encode(jsonStr);
        
        const formData = new FormData();
        formData.append('AppKey', 'B623D8449570');
        formData.append('ClientID', clientId);
        formData.append('ParamsData', base64Params);
        
        const apiPort = serverConfig.port.split(',')[0] || '5280';
        const url = `http://${serverConfig.ip}:${apiPort}/JeenpiAPI/V1/PlayMovieDirectly`;
        
        const response = await fetch(url, {
            method: 'POST',
            body: formData
        });
        
        const encodedResponse = await response.text();
        const data = JSON.parse(encodedResponse);
        
        playLoading.classList.add('hidden');
        playLoading.classList.remove('flex');
        
        if (data.code && data.code > 0) {
            let errorMsg = data.msg || `播放请求失败，错误代码: ${data.code}`;
            
            if (data.code === 404) {
                errorMsg = "影片文件不存在，无法播放！";
            } else if (data.code === 403) {
                errorMsg = "您没有权限播放此影片。";
            } else if (data.code >= 500) {
                errorMsg = "服务器错误，请稍后再试。";
            }
            
            throw new Error(errorMsg);
        }
        
        if (data.Data && data.Data.URLFileName) {
                let videoUrl = `http://${serverConfig.ip}:${serverConfig.port.split(',')[1] || '5281'}${data.Data.URLFileName}`;
                
                // 保存字幕URL（从Subtitles字段获取）
                if (data.Data.Subtitles) {
                    movieData.SubtitleURL = data.Data.Subtitles;
                }
                
                // 处理.strm文件
                videoUrl = await processStrmFile(videoUrl);
                
                // 直接使用内置播放器播放
                playWithInternalPlayer(videoUrl);
            } else {
                throw new Error('未获取到有效的播放链接，无法播放！');
            }
    } catch (error) {
        playLoading.classList.add('hidden');
        playLoading.classList.remove('flex');
        showErrorDialog('播放失败', error.message);
    }
}

async function playWithInternalPlayer(videoUrl) {
    videoPlayerContainer.classList.remove('hidden');
    playerLoading.classList.remove('hidden');
    
    movieData.currentVideoUrl = videoUrl;
    moviePlayer.src = videoUrl;
    
    // 移除内置播放器加载字幕URL的功能
    // 清除旧的字幕轨道，不再自动添加新的字幕轨道
    const oldTracks = moviePlayer.querySelectorAll('track');
    oldTracks.forEach(track => track.remove());
    
    moviePlayer.load();
    
    try {
        await moviePlayer.play();
    } catch (playError) {}
    
    videoPlayerContainer.scrollIntoView({ behavior: 'smooth' });
}



function formatDuration(seconds) {
    if (!seconds || isNaN(seconds)) return '未知时长';
    
    const totalSeconds = Math.floor(Number(seconds));
    const hours = Math.floor(totalSeconds / 3600);
    const minutes = Math.floor((totalSeconds % 3600) / 60);
    const secs = totalSeconds % 60;
    
    const formattedHours = hours.toString().padStart(2, '0');
    const formattedMinutes = minutes.toString().padStart(2, '0');
    const formattedSeconds = secs.toString().padStart(2, '0');
    
    return hours > 0 
        ? `${formattedHours}:${formattedMinutes}:${formattedSeconds}`
        : `${formattedMinutes}:${formattedSeconds}`;
}

const chineseNumbers = {
    '零': 0, '一': 1, '二': 2, '三': 3, '四': 4,
    '五': 5, '六': 6, '七': 7, '八': 8, '九': 9,
    '十': 10, '百': 100, '千': 1000, '万': 10000
};

function chineseToNumber(chinese) {
    if (!chinese || typeof chinese !== 'string') return 0;
    
    const numMatch = chinese.match(/\d+/);
    if (numMatch) return parseInt(numMatch[0], 10);
    
    let result = 0;
    let temp = 0;
    
    for (let i = 0; i < chinese.length; i++) {
        const char = chinese[i];
        const num = chineseNumbers[char];
        
        if (num === undefined) continue;
        
        if (num >= 10) {
            if (temp === 0) temp = 1;
            temp *= num;
            result += temp;
            temp = 0;
        } else {
            temp += num;
        }
    }
    
    result += temp;
    return result > 0 ? result : 0;
}

function extractSortNumber(str) {
    if (!str) return 0;
    
    const numMatch = str.match(/\d+/);
    if (numMatch) return parseInt(numMatch[0], 10);
    
    return chineseToNumber(str);
}

// 存储导演和演员的完整信息
let celebrityInfo = {};

// 全局状态变量，用于跟踪信息卡显示状态
let castDetailTimeout = null;
let currentCastItem = null;

// 获取导演和演员的信息
async function fetchCelebrityAvatars() {
    const directors = movieData.WebData?.directors || '';
    const casts = movieData.WebData?.casts || '';
    
    // 合并导演和演员列表，去重，支持逗号和顿号分隔
    const allCelebrities = [...new Set([
        ...directors.split(/[、,]/), 
        ...casts.split(/[、,]/)
    ])];
    
    // 过滤掉空字符串和未知导演/演员
    const validCelebrities = allCelebrities.filter(name => 
        name && name !== '未知导演' && name !== '未知演员'
    );
    
    if (validCelebrities.length === 0) {
        return;
    }
    
    try {
        // 对每个演员单独调用接口，因为一次只能获取一个演员的信息
        for (const name of validCelebrities) {
            // 构建GetFieldValue请求参数 - 一次只传递一个演员
            const paramsDataObj = [{
                "Method": "GetCastHeadImage",
                "CastName": name
            }];
            
            const jsonStr = JSON.stringify(paramsDataObj);
            const base64Params = base64Encode(jsonStr);
            
            const formData = new FormData();
            formData.append('AppKey', 'B623D8449570');
            formData.append('ClientID', clientId);
            formData.append('ParamsData', base64Params);
            
            const apiPort = serverConfig.port.split(',')[0] || '5280';
            const url = `http://${serverConfig.ip}:${apiPort}/JeenpiAPI/V1/GetFieldValue`;
            
            const response = await fetch(url, {
                method: 'POST',
                body: formData
            });
            
            const encodedResponse = await response.text();
            const data = JSON.parse(encodedResponse);
            
            if (data.code && data.code > 0) {
                console.warn(`获取演员${name}信息失败，错误代码: ${data.code}, 错误信息: ${data.msg}`);
                continue;
            }
            
            if (data.Data && Array.isArray(data.Data) && data.Data.length > 0) {
                const celebrity = data.Data[0];
                if (celebrity.Name) {
                    // 组合完整的头像URL，使用用户设置的服务端IP和端口，加上AppDocPath前缀
                    const resourcePort = serverConfig.port.split(',')[1] || '5281';
                    if (celebrity.UserHeadUrl) {
                        celebrity.fullAvatarUrl = `http://${serverConfig.ip}:${resourcePort}/AppDocPath${celebrity.UserHeadUrl}`;
                    } else if (celebrity.ImgUrl) {
                        celebrity.fullAvatarUrl = celebrity.ImgUrl;
                    }
                    // 保存完整的演员信息
                    celebrityInfo[celebrity.Name] = celebrity;
                }
            }
        }
        
    } catch (error) {
        console.warn('获取演员信息失败:', error);
    }
}

async function fetchCustomFieldValues() {
    const targetPid = movieData.PID;
    const targetStoreSite = movieData.FileData?.StoreSite;
        
    if (!targetPid || !targetStoreSite) {
        return;
    }
    
    try {
        const paramsDataObj = [
            {
                "Method": "GetCustomClassText",
                "PID": targetPid
            },
            {
                "Method": "GetStoreSiteCustomCaption",
                "StoreSite": targetStoreSite
            }
        ];
        
        const jsonStr = JSON.stringify(paramsDataObj);
        const base64Params = base64Encode(jsonStr);
        
        const formData = new FormData();
        formData.append('AppKey', 'B623D8449570');
        formData.append('ClientID', clientId);
        formData.append('ParamsData', base64Params);
        
        const apiPort = serverConfig.port.split(',')[0] || '5280';
        const url = `http://${serverConfig.ip}:${apiPort}/JeenpiAPI/V1/GetFieldValue`;
        
        const response = await fetch(url, {
            method: 'POST',
            body: formData
        });
        
        const encodedResponse = await response.text();
        const data = JSON.parse(encodedResponse);
        
        if (data.code && data.code > 0) {
            throw new Error(data.msg || `获取自定义字段失败，错误代码: ${data.code}`);
        }
        
        if (data.Data && Array.isArray(data.Data)) {
            data.Data.forEach(item => {
                if (item.Method === 'GetStoreSiteCustomCaption' && item.Value) {
                    customFieldData.storeSiteAlias = item.Value;
                } else if (item.Method === 'GetCustomClassText' && item.Items && item.Items.length > 0) {
                    customFieldData.customClassText = item.Items.map(i => i.Text || i.Value).join('、');
                }
            });
        }
        
    } catch (error) {}

}

async function fetchGroupItems() {
    if (!movieData || !movieData.FileData?.GroupID) {
        return;
    }
    
    try {
        const paramsDataObj = {
            "GroupID": movieData.FileData.GroupID
        };
        
        const jsonStr = JSON.stringify(paramsDataObj);
        const base64Params = base64Encode(jsonStr);
        
        const formData = new FormData();
        formData.append('AppKey', 'B623D8449570');
        formData.append('ClientID', clientId);
        formData.append('ParamsData', base64Params);
        
        const apiPort = serverConfig.port.split(',')[0] || '5280';
        const url = `http://${serverConfig.ip}:${apiPort}/JeenpiAPI/V1/GetGroupMovieItems`;
        
        const response = await fetch(url, {
            method: 'POST',
            body: formData
        });
        
        const responseText = await response.text();
        const data = JSON.parse(responseText);
        
        if (data.code && data.code > 0) {
            throw new Error(data.msg || `获取分组视频失败，错误代码: ${data.code}`);
        }
        
        if (data.Data && Array.isArray(data.Data)) {
            groupItems = data.Data;
            
            const groupKind = movieData.FileData?.GroupKind;
            if (groupKind === 0) {
                groupItems.sort((a, b) => {
                    const aFileName = a.FileData?.FileName || '';
                    const bFileName = b.FileData?.FileName || '';
                    return extractSortNumber(aFileName) - extractSortNumber(bFileName);
                });
            } else if (groupKind === 1) {
                groupItems.sort((a, b) => {
                    const aYear = parseInt(a.WebData?.year || '0', 10);
                    const bYear = parseInt(b.WebData?.year || '0', 10);
                    return aYear - bYear;
                });
            }
        }
        
    } catch (error) {
        throw error;
    }
}

function switchToGroupItem(index, playAfterSwitch = false) {
    if (!groupItems || index < 0 || index >= groupItems.length) {
        return;
    }
    
    showLoadingState();
    
    currentVideoIndex = index;
    movieData = groupItems[index];
    
    fetchCustomFieldValues().then(() => {
        renderMovieDetail();
        
        if (playAfterSwitch) {
            // 保持视频容器隐藏，让用户通过对话框选择播放方式
            videoPlayerContainer.classList.add('hidden');
            playMovie();
        } else {
            videoPlayerContainer.classList.add('hidden');
        }
    });
}

function base64Encode(str) {
    let escaped = '';
    for (let i = 0; i < str.length; i++) {
        const code = str.charCodeAt(i);
        if (code < 128) {
            escaped += str[i];
        } else {
            escaped += '\\u' + code.toString(16).padStart(4, '0');
        }
    }
    return btoa(escaped);
}

function base64Decode(encodedStr) {
    try {
        const decoded = atob(encodedStr);
        return decoded.replace(/\\u([0-9a-fA-F]{4})/g, (match, hex) => {
            return String.fromCharCode(parseInt(hex, 16));
        });
    } catch (error) {
        throw new Error('解析播放链接失败');
    }
}

// 处理.strm文件，调用接口获取实际视频URL和字幕URL
async function processStrmFile(videoUrl) {
    // 检查是否是.strm文件
    if (videoUrl.toLowerCase().endsWith('.strm')) {
        try {
            // 准备调用UrlFileReader接口的参数
            const paramsDataObj = {
                "URL": videoUrl
            };
            
            const jsonStr = JSON.stringify(paramsDataObj);
            const base64Params = base64Encode(jsonStr);
            
            const formData = new FormData();
            formData.append('AppKey', 'B623D8449570');
            formData.append('ClientID', clientId);
            formData.append('ParamsData', base64Params);
            
            const apiPort = serverConfig.port.split(',')[0] || '5280';
            const url = `http://${serverConfig.ip}:${apiPort}/JeenpiAPI/V1/UrlFileReader`;
            
            // 调用接口
            const response = await fetch(url, {
                method: 'POST',
                body: formData
            });
            
            const encodedResponse = await response.text();
            const data = JSON.parse(encodedResponse);
            
            if (data.code && data.code > 0) {
                throw new Error(data.msg || `读取.strm文件失败，错误代码: ${data.code}`);
            }
            
            // 解析返回的TextData
            if (data.Data && data.Data.TextData) {
                const lines = data.Data.TextData.split('\n');
                let actualVideoUrl = lines[0]?.trim() || '';
                let actualSubtitleUrl = lines[1]?.trim() || '';
                
                // 对URL进行解码
                try {
                    if (actualVideoUrl) {
                        actualVideoUrl = decodeURIComponent(actualVideoUrl);
                    }
                    if (actualSubtitleUrl) {
                        actualSubtitleUrl = decodeURIComponent(actualSubtitleUrl);
                    }
                } catch (decodeError) {
                    console.warn('URL解码失败，使用原始URL:', decodeError);
                    // 如果解码失败，继续使用原始URL
                }
                
                if (!actualVideoUrl) {
                    throw new Error('从.strm文件中未获取到有效的视频URL');
                }
                
                // 如果有字幕URL，更新movieData中的字幕信息
                if (actualSubtitleUrl) {
                    movieData.SubtitleURL = actualSubtitleUrl;
                }
                
                return actualVideoUrl;
            } else {
                throw new Error('从.strm文件中未获取到有效数据');
            }
        } catch (error) {
            console.error('处理.strm文件时出错:', error);
            throw error;
        }
    }
    
    // 不是.strm文件，直接返回原始URL
    return videoUrl;
}

async function fetchMovieDetailFromApi(pid, autoPlay = false) {
    showLoadingState();
    
    try {
        const formData = new FormData();
        formData.append('AppKey', 'B623D8449570');
        formData.append('ClientID', clientId);
        
        const paramsDataObj = {
            "PID": pid
        };
        formData.append('ParamsData', JSON.stringify(paramsDataObj));
        
        const apiPort = serverConfig.port.split(',')[0] || '5280';
        const url = `http://${serverConfig.ip}:${apiPort}/JeenpiAPI/V1/GetMovieDetail`;
        
        const response = await fetch(url, {
            method: 'POST',
            body: formData
        });
        
        const data = await response.json();
        
        if (data.code > 0) {
            throw new Error(data.msg || `获取影片详情失败，错误代码: ${data.code}`);
        }
        
        if (data.Data) {
            movieData = data.Data;
            await fetchCustomFieldValues();
            // 获取导演和演员头像
            await fetchCelebrityAvatars();
            
            if (movieData.FileData?.GroupID && movieData.FileData.GroupID !== '') {
                await fetchGroupItems();
                if (groupItems.length > 0) {
                    currentVideoIndex = 0;
                    movieData = groupItems[0];
                    await fetchCustomFieldValues();
                    // 重新获取导演和演员头像
                    await fetchCelebrityAvatars();
                }
            }
            
            renderMovieDetail();
            
            // 如果需要自动播放，在渲染完成后触发播放
            if (autoPlay) {
                setTimeout(() => {
                    const playBtn = document.getElementById('playBtn');
                    if (playBtn) {
                        playBtn.click();
                    }
                }, 100); // 短暂延迟确保DOM已完全渲染
            }
        } else {
            throw new Error('未找到影片详情数据');
        }
    } catch (error) {
        showErrorState();
    }
}

function generateRatingHtml(rating) {
    const score = Math.max(0, Math.min(10, parseFloat(rating) || 0));
    const fullStars = Math.floor(score / 2);
    const hasHalfStar = score % 2 >= 1;
    
    let starsHtml = '';
    
    for (let i = 0; i < fullStars; i++) {
        starsHtml += '<i class="fa fa-star text-rating"></i>';
    }
    
    if (hasHalfStar) {
        starsHtml += '<i class="fa fa-star-half-o text-rating"></i>';
    }
    
    const emptyStars = 5 - fullStars - (hasHalfStar ? 1 : 0);
    for (let i = 0; i < emptyStars; i++) {
        starsHtml += '<i class="fa fa-star-o text-rating"></i>';
    }
    
    return `
        <!-- 移动端水平布局，桌面端垂直布局 -->
        <div class="bg-yellow-50 p-3 sm:p-4 rounded-lg">
            <!-- 移动端水平布局 -->
            <div class="flex items-center justify-between sm:flex-col sm:items-center w-full">
                <div class="flex items-center">
                    <div class="text-2xl sm:text-3xl font-bold text-gray-900 mr-3">${score.toFixed(1)}</div>
                    <div class="flex">${starsHtml}</div>
                </div>
                <div class="hidden sm:block text-sm text-gray-500 my-1">影片评分</div>
                <div class="w-full mt-2 sm:mt-1 rating-bar">
                    <div class="rating-fill" style="width: ${score * 10}%"></div>
                </div>
            </div>
        </div>
    `;
}

function formatFileSize(bytes) {
    if (bytes === 0 || !bytes) return '未知大小';
    
    const units = ['B', 'KB', 'MB', 'GB', 'TB'];
    let unitIndex = 0;
    
    while (bytes >= 1024 && unitIndex < units.length - 1) {
        bytes /= 1024;
        unitIndex++;
    }
    
    return `${bytes.toFixed(2)} ${units[unitIndex]}`;
}

function mapQuality(qualityValue) {
    switch(qualityValue) {
        case 0: return 'Low';
        case 1: return '480P';
        case 2: return '720P';
        case 3: return '1080P';
        case 4: return '4K';
        default: return '未知';
    }
}

function limitAliases(aliases, maxCount = 3) {
    if (!aliases || !Array.isArray(aliases)) return [];
    
    if (typeof aliases === 'string') {
        aliases = aliases.split(',').map(a => a.trim()).filter(a => a);
    }
    
    return aliases.slice(0, maxCount);
}

function renderMovieDetail() {
    if (!movieData) {
        showErrorState();
        return;
    }
    
    const displayData = movieData;
    
    let posterUrl = displayData.PosterURL;
    const resourcePort = serverConfig.port.split(',')[1] || '5281';
    
    if (posterUrl && !posterUrl.startsWith('http')) {
        posterUrl = `http://${serverConfig.ip}:${resourcePort}${posterUrl}`;
    } else if (!posterUrl) {
        posterUrl = `https://picsum.photos/seed/${displayData.PID || 'default'}/800/1200`;
    }
    
    // 对于电视剧，使用GroupID作为标题
    const title = displayData.FileData?.GroupID && displayData.FileData.GroupID !== '' ? 
                  displayData.FileData.GroupID : 
                  (displayData.WebData?.title || displayData.FileData?.Caption || '未知影片');
    const originalTitle = displayData.WebData?.original_title || '';
    
    const aliases = limitAliases(displayData.WebData?.aliases);
    const aliasesHtml = aliases.length > 0 
        ? `<p class="text-gray-500 mt-1 truncate" title="别名: ${aliases.join(' / ')}">别名: ${aliases.join(' / ')}</p>` 
        : '';
        
    const year = displayData.WebData?.year || '未知年份';
    const countries = displayData.WebData?.countries || '';
    const genres = displayData.WebData?.genres || '';
    const directors = displayData.WebData?.directors || '未知导演';
    const casts = displayData.WebData?.casts || '未知演员';
    const rating = displayData.WebData?.rating_douban || '暂无评分';
    const summary = displayData.WebData?.summary || '暂无剧情简介';
    
    const durationSeconds = displayData.FileData?.Duration;
    const duration = formatDuration(durationSeconds);
    
    const ratingHtml = generateRatingHtml(rating);
    
    const sizeBytes = movieData.FileData?.Size;
    const size = formatFileSize(sizeBytes);
    const videoFormat = movieData.VideoFormat || '未知';
    const audioFormat = movieData.AudioFormat || '未知';
    const qualityValue = movieData.Quality;
    const quality = mapQuality(qualityValue);
    const fileLocation = displayData.FileData?.StoreSite || '未知位置';
    const displayFileLocation = customFieldData.storeSiteAlias || fileLocation;
    
    // 检查是否有字幕信息
    const hasSubtitle = displayData.Subtitles || displayData.SubtitleURL || movieData.SubtitleURL;
    const subtitleHtml = hasSubtitle 
        ? `<div class="bg-white p-4 rounded-lg shadow-sm border border-gray-100">
            <h4 class="text-sm font-medium text-gray-500 mb-2">字幕</h4>
            <p class="text-gray-800 flex items-center">
                <i class="fa fa-check-circle text-green-500 mr-2"></i>有字幕文件
            </p>
          </div>`
        : '';
    
    const labelsHtml = displayData.Labels && displayData.Labels.length > 0 
        ? `<div class="mt-8">
            <h2 class="text-xl font-semibold text-gray-800 mb-3 flex items-center">
                <i class="fa fa-tags mr-2 text-primary"></i>标签
            </h2>
            <div class="flex flex-wrap gap-2">
                ${displayData.Labels.split(',').map(label => 
                    `<span class="bg-blue-50 text-blue-700 px-3 py-1 rounded text-sm">${label}</span>`
                ).join('')}
            </div>
          </div>` 
        : '';
    
    const customClassHtml = customFieldData.customClassText 
        ? `<div class="mt-8">
            <h2 class="text-xl font-semibold text-gray-800 mb-3 flex items-center">
                <i class="fa fa-cubes mr-2 text-primary"></i>自定义类别
            </h2>
            <div class="bg-white p-4 rounded-lg shadow-sm border border-gray-100">
                <p class="text-gray-700">${customFieldData.customClassText}</p>
            </div>
          </div>` 
        : '';
    
    let groupItemsHtml = '';
    const groupKind = movieData.FileData?.GroupKind;
    
    if (groupItems && groupItems.length > 0) {
        if (groupKind === 0) {
            groupItemsHtml = `<div class="mt-8">
                <h2 class="text-xl font-semibold text-gray-800 mb-4 flex items-center">
                    <i class="fa fa-television mr-2 text-primary"></i>剧集列表
                </h2>
                <div class="flex flex-wrap gap-2 max-h-40 overflow-y-auto pb-2">
                    ${groupItems.map((item, index) => {
                        const displayName = item.FileData?.SubCaptionByGroup || 
                                          `${index + 1}集`;
                        
                        return `
                        <button 
                            class="episode-btn ${index === currentVideoIndex ? 'episode-btn-active' : 'episode-btn-inactive'}"
                            onclick="switchToGroupItem(${index}, true)"
                        >
                            ${displayName}
                        </button>
                    `;
                    }).join('')}
                </div>
            </div>`;
        } else if (groupKind === 1) {
            groupItemsHtml = `<div class="mt-8">
                <h2 class="text-xl font-semibold text-gray-800 mb-4 flex items-center">
                    <i class="fa fa-film mr-2 text-primary"></i>系列影片
                </h2>
                <div class="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 lg:grid-cols-5 gap-4">
                    ${groupItems.map((item, index) => {
                        let itemPosterUrl = item.PosterURL;
                        if (itemPosterUrl && !itemPosterUrl.startsWith('http')) {
                            itemPosterUrl = `http://${serverConfig.ip}:${resourcePort}${itemPosterUrl}`;
                        } else if (!itemPosterUrl) {
                            itemPosterUrl = `https://picsum.photos/seed/${item.PID}/300/450`;
                        }
                        
                        const itemTitle = item.FileData?.SubCaptionByGroup || 
                                        item.WebData?.title || 
                                        item.FileData?.Caption || 
                                        '未知影片';
                        const itemYear = item.WebData?.year || '未知年份';
                        
                        return `
                            <div 
                                class="group cursor-pointer"
                                onclick="switchToGroupItem(${index}, false)"
                            >
                                <div class="relative rounded overflow-hidden shadow-sm">
                                    <img src="${itemPosterUrl}" alt="${itemTitle}的海报" class="w-full aspect-[2/3] object-cover transition-transform duration-300 group-hover:scale-105">
                                    ${index === currentVideoIndex ? `
                                        <div class="absolute top-2 right-2 bg-primary text-white text-xs px-2 py-1 rounded-full">
                                            当前
                                        </div>
                                    ` : ''}
                                </div>
                                <p class="mt-2 text-sm font-medium text-gray-800 truncate">${itemTitle}</p>
                                <p class="text-xs text-gray-500">${itemYear}</p>
                            </div>
                        `;
                    }).join('')}
                </div>
            </div>`;
        }
    }
    
    movieDetailContent.innerHTML = `
        <div class="md:flex gap-6">
            <div class="md:w-1/3 lg:w-1/4 mb-10 md:mb-0">
                <div class="sticky top-20">
                    <img src="${posterUrl}" alt="${title}的海报" class="w-full h-auto object-cover rounded-lg shadow-md">
                    
                    <div class="mt-6 space-y-4">
                        <button id="playBtn" class="w-full bg-primary hover:bg-primary/90 text-white font-medium py-3 px-4 rounded-lg transition-all duration-200 flex items-center justify-center">
                            <i class="fa fa-play mr-2"></i> 立即播放
                        </button>
                    </div>
                </div>
            </div>
            
            <div class="md:w-2/3 lg:w-3/4 pt-0 md:pt-0">
                <div class="flex flex-wrap gap-4">
                    <div class="flex-1 min-w-0">
                        <h1 class="text-2xl md:text-3xl font-bold text-gray-900 truncate">${title}</h1>
                        ${originalTitle ? `<p class="text-gray-500 mt-1 truncate" title="${originalTitle}">${originalTitle}</p>` : ''}
                        ${aliasesHtml}
                        <p class="text-gray-600 mt-2">${year} / ${countries} / ${duration}</p>
                    </div>
                    <div class="shrink-0">
                        ${ratingHtml}
                    </div>
                </div>
                
                <div class="mt-4 flex flex-wrap gap-2">
                    ${genres.split(',').map(genre => 
                        `<span class="bg-gray-100 text-gray-700 px-3 py-1 rounded text-sm">${genre}</span>`
                    ).join('')}
                </div>
                
                <div class="mt-8">
                    <h2 class="text-xl font-semibold text-gray-800 mb-3 flex items-center">
                        <i class="fa fa-info-circle mr-2 text-primary"></i>剧情简介
                    </h2>
                    <div class="bg-white p-4 rounded-lg shadow-sm border border-gray-100">
                        <p class="text-gray-700 leading-relaxed">${summary}</p>
                    </div>
                </div>
                
                <div class="mt-6 grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div>
                        <h3 class="text-lg font-semibold text-gray-800 mb-2">导演</h3>
                        <div class="flex flex-wrap gap-6 items-center justify-center md:justify-start">
                            ${(() => {
                                // 支持逗号和顿号分隔
                                const directorList = directors.split(/[、,]/).filter(name => name.trim() !== '');
                                
                                return directorList
                                    .map(director => {
                                        const avatarUrl = celebrityInfo[director]?.fullAvatarUrl;
                                        return `
                                            <div class="flex items-center gap-3 cast-item" data-cast-name="${director}" data-cast-role="导演">
                                                ${avatarUrl ? `
                                                <div class="w-14 h-14 rounded-lg overflow-hidden border-2 border-gray-200 hover:border-primary transition-all duration-200">
                                                    <img src="${avatarUrl}" alt="${director}" class="w-full h-full object-cover hover:scale-105 transition-transform duration-200">
                                                </div>` : ''}
                                                <span class="text-gray-700 font-normal whitespace-nowrap">${director}</span>
                                            </div>
                                        `;
                                    }).join('');
                                })()}
                        </div>
                    </div>
                    <div>
                        <h3 class="text-lg font-semibold text-gray-800 mb-2">主演</h3>
                        ${(() => {
                            // 支持逗号和顿号分隔
                            const castList = casts.split(/[、,]/).filter(name => name.trim() !== '');
                            // 根据页面宽度动态显示演员数量
                            let maxCasts = 10; // 默认显示10个
                            const windowWidth = window.innerWidth;
                            if (windowWidth < 640) maxCasts = 5; // 移动端显示5个
                            else if (windowWidth < 1024) maxCasts = 8; // 平板显示8个
                            
                            // 只显示前maxCasts个演员
                            const displayedCasts = castList.slice(0, maxCasts);
                            
                            // 检查是否有演员获取到完整信息
                            const hasValidCasts = displayedCasts.some(cast => celebrityInfo[cast] && celebrityInfo[cast].fullAvatarUrl);
                            
                            if (!hasValidCasts) {
                                // 如果没有演员获取到完整信息，显示逗号分隔的名字列表
                                return `<div class="bg-white p-4 rounded-lg shadow-sm border border-gray-100">
                                    <p class="text-gray-700">${displayedCasts.join('、')}</p>
                                </div>`;
                            } else {
                                // 否则显示正常的演员列表
                                return `<div class="relative pb-4">
                                    <div class="absolute inset-y-0 left-0 flex items-center z-20">
                                        <button class="ml-2 bg-white rounded-full p-1 shadow-md hover:bg-gray-50 transition-colors flex items-center justify-center" style="width: 28px; height: 28px; display: none;" onclick="scrollHorizontal('castsScroll', -1)" title="向左滚动">
                                            <i class="fa fa-chevron-left text-gray-600 text-sm"></i>
                                        </button>
                                    </div>
                                    <div class="flex overflow-x-auto pb-2 gap-4 scroll-hidden pl-0 pr-0" id="castsScroll" style="padding-left: var(--scroll-padding-left, 0px); padding-right: var(--scroll-padding-right, 0px);">
                                        ${displayedCasts
                                            .filter(cast => celebrityInfo[cast] && celebrityInfo[cast].fullAvatarUrl) // 只显示有完整信息的演员
                                            .map(cast => {
                                                const avatarUrl = celebrityInfo[cast]?.fullAvatarUrl;
                                                return `
                                                    <div class="flex flex-col items-center min-w-18 flex-shrink-0 cast-item" data-cast-name="${cast}" data-cast-role="演员">
                                                        <div class="w-14 h-14 rounded-lg overflow-hidden border-2 border-gray-200 hover:border-primary transition-all duration-200">
                                                            <img src="${avatarUrl}" alt="${cast}" class="w-full h-full object-cover hover:scale-105 transition-transform duration-200">
                                                        </div>
                                                        <span class="text-sm text-gray-700 font-normal mt-1 truncate text-center max-w-18 whitespace-nowrap">${cast}</span>
                                                    </div>
                                                `;
                                            }).join('')}
                                    </div>
                                    <div class="absolute inset-y-0 right-0 flex items-center z-20">
                                        <button class="mr-2 bg-white rounded-full p-1 shadow-md hover:bg-gray-50 transition-colors flex items-center justify-center" style="width: 28px; height: 28px; display: none;" onclick="scrollHorizontal('castsScroll', 1)" title="向右滚动">
                                            <i class="fa fa-chevron-right text-gray-600 text-sm"></i>
                                        </button>
                                    </div>
                                </div>`;
                            }
                        })()}
                    </div>
                </div>
                
                ${customClassHtml}
                ${labelsHtml}
                ${groupItemsHtml}
                
                <div class="mt-8">
                    <h3 class="text-lg font-semibold text-gray-800 mb-4">文件信息</h3>
                    <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4">
                        <div class="bg-white p-4 rounded-lg shadow-sm border border-gray-100">
                            <h4 class="text-sm font-medium text-gray-500 mb-2">视频格式</h4>
                            <p class="text-gray-800">${videoFormat} (${quality})</p>
                        </div>
                        <div class="bg-white p-4 rounded-lg shadow-sm border border-gray-100">
                            <h4 class="text-sm font-medium text-gray-500 mb-2">音频格式</h4>
                            <p class="text-gray-800">${audioFormat}</p>
                        </div>
                        <div class="bg-white p-4 rounded-lg shadow-sm border border-gray-100">
                            <h4 class="text-sm font-medium text-gray-500 mb-2">文件大小</h4>
                            <p class="text-gray-800">${size}</p>
                        </div>
                        <div class="bg-white p-4 rounded-lg shadow-sm border border-gray-100">
                            <h4 class="text-sm font-medium text-gray-500 mb-2">存储位置</h4>
                            <p class="text-gray-800 truncate" title="${fileLocation}">${displayFileLocation}</p>
                        </div>
                    </div>
                    ${hasSubtitle ? `
                    <div class="mt-4">
                        <div class="bg-green-50 border border-green-200 p-4 rounded-lg flex items-center">
                            <i class="fa fa-file-text-o text-green-600 text-2xl mr-3"></i>
                            <div>
                                <h4 class="text-sm font-semibold text-green-800 mb-1">字幕文件</h4>
                                <p class="text-xs text-green-600">此影片包含字幕文件，支持外部播放器和内置播放器</p>
                            </div>
                        </div>
                    </div>
                    ` : ''}
                </div>
            </div>
        </div>
    `;
    
    document.getElementById('playBtn').addEventListener('click', playMovie);
    
    // 添加演员悬停事件监听器
    setTimeout(() => {
        addCastHoverListeners();
        // 检查并设置滚动按钮的显示状态
        checkScrollButtons();
        // 添加窗口大小变化事件监听
        window.addEventListener('resize', checkScrollButtons);
    }, 100);
    
    loadingIndicator.classList.add('hidden');
    errorState.classList.add('hidden');
    movieDetailContent.classList.remove('hidden');
}

// 检查并设置滚动按钮的显示状态
function checkScrollButtons() {
    const castsScroll = document.getElementById('castsScroll');
    if (castsScroll) {
        const leftBtn = castsScroll.previousElementSibling?.querySelector('button');
        const rightBtn = castsScroll.nextElementSibling?.querySelector('button');
        
        if (castsScroll.scrollWidth > castsScroll.clientWidth) {
            // 需要滚动，显示按钮
            if (leftBtn) leftBtn.style.display = 'flex';
            if (rightBtn) rightBtn.style.display = 'flex';
            // 设置内边距，避免内容被按钮遮挡
            castsScroll.style.paddingLeft = '4rem';
            castsScroll.style.paddingRight = '4rem';
        } else {
            // 不需要滚动，隐藏按钮
            if (leftBtn) leftBtn.style.display = 'none';
            if (rightBtn) rightBtn.style.display = 'none';
            // 移除内边距，让列表与标题对齐
            castsScroll.style.paddingLeft = '0';
            castsScroll.style.paddingRight = '0';
        }
    }
}

// 添加事件委托处理演员悬停
function addCastHoverListeners() {
    const castItems = document.querySelectorAll('.cast-item');
    const castDetail = document.getElementById('castDetail');
    
    // 确保所有cast-item都能正确触发事件
    castItems.forEach(item => {
        // 鼠标进入事件 - 使用捕获阶段确保事件能被正确触发
        item.addEventListener('mouseenter', (e) => {
            // 清除任何待处理的隐藏定时器
            clearTimeout(castDetailTimeout);
            
            const castName = item.dataset.castName || item.querySelector('span')?.textContent?.trim();
            const role = item.dataset.castRole || (item.closest('div')?.querySelector('h3')?.textContent?.includes('导演') ? '导演' : '演员');
            if (castName) {
                showCastDetail(e, castName, role);
                currentCastItem = item;
            }
        }, true);
        
        // 鼠标离开事件
        item.addEventListener('mouseleave', (e) => {
            // 确保事件目标正确
            if (e.target === item || item.contains(e.target)) {
                // 设置延迟隐藏，给用户时间移动到信息卡上
                castDetailTimeout = setTimeout(() => {
                    // 检查鼠标是否在任何演员项或信息卡上
                    const mouseX = e.clientX;
                    const mouseY = e.clientY;
                    const mouseOverElement = document.elementFromPoint(mouseX, mouseY);
                    
                    // 检查鼠标是否在任何演员项上
                    const isOverCastItem = Array.from(castItems).some(castItem => castItem.contains(mouseOverElement));
                    // 检查鼠标是否在信息卡上
                    const isOverCastDetail = castDetail.contains(mouseOverElement);
                    
                    if (!isOverCastItem && !isOverCastDetail) {
                        hideCastDetail();
                        currentCastItem = null;
                    }
                }, 200);
            }
        });
    });
    
    // 确保弹窗本身不会触发隐藏
    castDetail.addEventListener('mouseenter', () => {
        // 清除任何待处理的隐藏定时器
        clearTimeout(castDetailTimeout);
    });
    
    castDetail.addEventListener('mouseleave', (e) => {
        // 设置延迟隐藏，给用户时间移回演员项
        castDetailTimeout = setTimeout(() => {
            // 检查鼠标是否在任何演员项上
            const mouseX = e.clientX;
            const mouseY = e.clientY;
            const mouseOverElement = document.elementFromPoint(mouseX, mouseY);
            const isOverCastItem = Array.from(castItems).some(castItem => castItem.contains(mouseOverElement));
            
            if (!isOverCastItem) {
                hideCastDetail();
                currentCastItem = null;
            }
        }, 200);
    });
}

// 水平滚动函数
function scrollHorizontal(elementId, direction) {
    const element = document.getElementById(elementId);
    if (element) {
        const scrollAmount = 200; // 每次滚动的距离
        element.scrollBy({ left: scrollAmount * direction, behavior: 'smooth' });
    }
}

// 显示演员详细信息
function showCastDetail(event, castName, role) {
    const castDetail = document.getElementById('castDetail');
    const castDetailName = document.getElementById('castDetailName');
    const castDetailRole = document.getElementById('castDetailRole');
    const castDetailAvatar = document.getElementById('castDetailAvatar');
    const castDetailBio = document.getElementById('castDetailBio');
    const castDetailSex = document.getElementById('castDetailSex');
    const castDetailXingzuo = document.getElementById('castDetailXingzuo');
    const castDetailBirthday = document.getElementById('castDetailBirthday');
    const castDetailBirthplace = document.getElementById('castDetailBirthplace');
    const castDetailProfession = document.getElementById('castDetailProfession');
    
    // 获取演员完整信息
    const celebrity = celebrityInfo[castName];
    
    // 设置基本信息
    castDetailName.textContent = castName;
    castDetailRole.textContent = role;
    
    // 设置头像
    const avatarUrl = celebrity?.fullAvatarUrl;
    const avatarImg = castDetailAvatar.querySelector('img');
    if (avatarUrl) {
        avatarImg.src = avatarUrl;
        avatarImg.alt = castName;
        castDetailAvatar.classList.remove('hidden');
    } else {
        castDetailAvatar.classList.add('hidden');
    }
    
    // 设置详细信息
    if (celebrity) {
        castDetailSex.textContent = celebrity.Sex || '未知';
        castDetailXingzuo.textContent = celebrity.Xingzuo || '未知';
        castDetailBirthday.textContent = celebrity.Birthday ? formatBirthday(celebrity.Birthday) : '未知';
        castDetailBirthplace.textContent = celebrity.Birthplace || '未知';
        castDetailProfession.textContent = celebrity.Profession || '未知';
        castDetailBio.textContent = celebrity.Introduction || `${castName}是一位${role}，暂无更多详细信息。`;
    } else {
        // 默认信息
        castDetailSex.textContent = '未知';
        castDetailXingzuo.textContent = '未知';
        castDetailBirthday.textContent = '未知';
        castDetailBirthplace.textContent = '未知';
        castDetailProfession.textContent = '未知';
        castDetailBio.textContent = `${castName}是一位${role}，暂无更多详细信息。`;
    }
    
    // 计算弹出位置
    const rect = event.currentTarget.getBoundingClientRect();
    const popupWidth = castDetail.offsetWidth;
    const popupHeight = castDetail.offsetHeight;
    const windowHeight = window.innerHeight;
    
    // 确保弹窗在可视区域内
    let left = rect.left + rect.width / 2 - popupWidth / 2;
    let top = rect.bottom + 10; // 默认显示在头像下方
    
    // 调整水平位置
    if (left < 10) left = 10;
    if (left + popupWidth > window.innerWidth - 10) left = window.innerWidth - popupWidth - 10;
    
    // 调整垂直位置
    if (top + popupHeight > windowHeight - 10) {
        // 如果下方超出屏幕，显示在头像上方
        top = rect.top - popupHeight - 10;
    }
    
    // 确保顶部不超出屏幕
    if (top < 10) top = 10;
    
    // 设置弹窗位置
    castDetail.style.left = `${left}px`;
    castDetail.style.top = `${top}px`;
    castDetail.style.display = 'block';
    castDetail.style.pointerEvents = 'auto';
    
    // 强制重绘
    castDetail.offsetHeight;
    
    // 添加淡入效果
    castDetail.classList.remove('opacity-0');
    castDetail.classList.add('opacity-100');
}

// 格式化生日（支持多种格式：8位数字格式和Delphi TDateTime类型）
function formatBirthday(birthdayValue) {
    if (!birthdayValue) return '未知';
    
    // 转换为字符串进行处理
    const birthdayStr = String(birthdayValue);
    
    // 检查是否为8位数字格式 (YYYYMMDD)
    if (birthdayStr.length === 8 && /^\d+$/.test(birthdayStr)) {
        const year = birthdayStr.substring(0, 4);
        const month = birthdayStr.substring(4, 6);
        const day = birthdayStr.substring(6, 8);
        return `${year}-${month}-${day}`;
    }
    
    // 检查是否为Delphi TDateTime格式（浮点数）
    const tDateTime = parseFloat(birthdayStr);
    if (!isNaN(tDateTime)) {
        // Delphi TDateTime是自1899-12-30以来的天数
        const baseDate = new Date(1899, 11, 30); // 注意：JavaScript月份从0开始，11表示12月
        const days = Math.floor(tDateTime);
        
        // 计算目标日期
        const targetDate = new Date(baseDate);
        targetDate.setDate(baseDate.getDate() + days);
        
        // 格式化日期为YYYY-MM-DD
        return targetDate.toISOString().split('T')[0];
    }
    
    // 如果无法识别格式，返回原始值
    return birthdayStr;
}

// 隐藏演员详细信息
function hideCastDetail() {
    const castDetail = document.getElementById('castDetail');
    castDetail.classList.remove('opacity-100');
    castDetail.classList.add('opacity-0');
    
    setTimeout(() => {
        castDetail.style.display = 'none';
        castDetail.style.pointerEvents = 'none';
    }, 300);
}

function showLoadingState() {
    loadingIndicator.classList.remove('hidden');
    errorState.classList.add('hidden');
    movieDetailContent.classList.add('hidden');
}

function showErrorState() {
    loadingIndicator.classList.add('hidden');
    errorState.classList.remove('hidden');
    movieDetailContent.classList.add('hidden');
}

document.addEventListener('DOMContentLoaded', initPage);